var serial = {};

(function() {
  'use strict';

  // Tableau global pour suivre les appareils connectés
  serial.connectedDevices = [];

  serial.getPorts = function() {
    return navigator.usb.getDevices().then(devices => {
      return devices.map(device => new serial.Port(device));
    });
  };

  serial.requestPort = function() {
    const filters = [
      //{ 'vendorId': 0x239A }, // Adafruit boards
      { 'vendorId': 0x2e8a, 'productId': 0x0005 }, // Raspberry Pi
      // { 'vendorId': 0xcafe }, // TinyUSB example
      // { 'vendorId': 0x1209, 'productId': 0xADDA }, // MicroPython boards
    ];
    return navigator.usb.requestDevice({ 'filters': filters }).then(
      device => new serial.Port(device)
    );
  }

  serial.Port = function(device) {
    this.device_ = device;
    this.interfaceNumber = 0;
    this.endpointIn = 0;
    this.endpointOut = 0;
    this.id = null; // Ajoutez un champ pour l'ID de l'appareil
    this.reading = false; // Propriété pour indiquer l'état de la lecture
  };

 // Méthode connect pour initialiser la connexion et démarrer la lecture
serial.Port.prototype.connect = function() {
  let readLoop = () => {
      if (!this.reading) {
          return; // Arrêter la lecture si la propriété reading est false
      }

      this.device_.transferIn(this.endpointIn, 64).then(result => {
          this.onReceive(result.data);
          readLoop();
      }, error => {
          this.onReceiveError(error);
      });
  };

  return this.device_.open()
      .then(() => {
          if (this.device_.configuration === null) {
              return this.device_.selectConfiguration(1);
          }
      })
      .then(() => {
          var interfaces = this.device_.configuration.interfaces;
          interfaces.forEach(element => {
              element.alternates.forEach(elementalt => {
                  if (elementalt.interfaceClass === 0xFF) {
                      this.interfaceNumber = element.interfaceNumber;
                      elementalt.endpoints.forEach(elementendpoint => {
                          if (elementendpoint.direction === "out") {
                              this.endpointOut = elementendpoint.endpointNumber;
                          }
                          if (elementendpoint.direction === "in") {
                              this.endpointIn = elementendpoint.endpointNumber;
                          }
                      });
                  }
              });
          });
      })
      .then(() => this.device_.claimInterface(this.interfaceNumber))
      .then(() => this.device_.selectAlternateInterface(this.interfaceNumber, 0))
      .then(() => this.device_.controlTransferOut({
          'requestType': 'class',
          'recipient': 'interface',
          'request': 0x22,
          'value': 0x01,
          'index': this.interfaceNumber
      }))
      .then(() => {
          serial.connectedDevices.push(this); // Ajouter l'appareil à la liste des appareils connectés
          this.reading = true; // Démarrer la lecture
          readLoop();
      });
};

// Méthode pour arrêter la lecture
serial.Port.prototype.stopReading = function() {
  this.reading = false;
};

  // Méthode pour déconnecter le périphérique
serial.Port.prototype.disconnect = function() {
  return this.device_.controlTransferOut({
          'requestType': 'class',
          'recipient': 'interface',
          'request': 0x22,
          'value': 0x00,
          'index': this.interfaceNumber
      })
      .then(() => {
          this.stopReading(); // Arrêter la lecture avant de fermer le périphérique
          this.device_.close();
          serial.connectedDevices = serial.connectedDevices.filter(d => d !== this); // Retirer l'appareil de la liste des appareils connectés
      });
};

  serial.Port.prototype.send = function(data) {
    return this.device_.transferOut(this.endpointOut, data);
  };
})();